from Mathutils import Matrix, Vector
from math import fabs
import GameLogic
from MyGameTimer import MY_SingleTimer


class MY_SingleGameWorld(object):
	"""
	singleton class of the game's world to set all its laws
	
	@ivar gravityForce: the power of the gravity (default 9.81)
	@type gravityForce: Float
	@ivar gravityVector: the standardized vector for gravity direction
	@type gravityVector: Vector
	@ivar planeForceDirection: the gravity direction in the current plane (up=1 or down=-1)
	@type planeForceDirection: Integer (1 or -1)
	@ivar isPaused: True if the whole world was stopped by an event
	@type isPaused: Boolean
	@ivar gravityAxis: the axis of the gravity direction
	@type gravityAxis: Integer (0,1,2)
	@ivar nextItemTime: this is the counter variable for setting an item in the game
	@type nextItemTime: Integer
	@ivar nextHurdle: True if it is time for a new hurdle
	@type nextHurdle: Boolean
	@ivar specialKeys: True if the player has to use special key input dependent on the current plane of the player
	@type specialKeys: Boolean
	@ivar numberOfHurdle: the number of all hurdles in the game
	@type numberOfHurdle: Integer
	@ivar deleteDelay: the amount of the delay for deleting objects
	@type deleteDelay: Integer
	@ivar boxIntro: True if player has to use the power switch to start the game
	@type boxIntro: Boolean
	@ivar soundOn: True if the sound in the background is switched on, False for off
	@type soundOn: Boolean
	@ivar switchOn: True is player has used the power switch
	@type switchOn: Boolean
	@ivar restartGame: True if the game shall be restarted
	@type restartGame: Boolean
	@ivar tpcOn: True if the third person camera shall appear in the lower left corner of the window, False if you wish no visible third person camera
	@type tpcOn: Boolean
	@ivar planeRotationMatrix: the rotation matrix that holds the information for every object to know how to get the head up in the sky (world coords)
	@type planeRotationMatrix: Matrix
	@ivar isUnusual: True if you want the pure stair insanity
	@type isUnusual: Boolean
	"""
	
	__instance = None  # the unique instance

	def __new__(self, *args, **kargs):
		if self.__instance is None:
			self.__instance = object.__new__(self, *args, **kargs)
			
			# __init__ section
			self.gravityForce = 9.81
			self.gravityVector = Vector([0, 0, -1])
			self.planeForceDirection = -1
			self.isPaused = True
			self.gravityAxis = 2
			self.nextItemTime = 0
			self.nextHurdle = False
			self.specialKeys = False
			self.numberOfHurdle = 0
			self.deleteDelay = 0
			self.boxIntro = True
			self.soundOn = True
			self.switchOn = False
			self.restartGame = False
			self.tpcOn = False
			self.planeRotationMatrix = 0
			self.isUnusual = False
			
		return self.__instance

	def __init__(self):
		pass
	
	def setDefaultWorld(self):
		"""
		resets the whole game world to default values
		"""
		
		self.gravityForce = 9.81
		self.gravityVector = Vector([0, 0, -1])
		self.planeForceDirection = -1
		self.isPaused = True
		self.gravityAxis = 2
		self.nextItemTime = 0
		self.nextHurdle = False
		self.specialKeys = False
		self.numberOfHurdle = 0
		self.deleteDelay = 0
		self.boxIntro = True
		self.soundOn = True
		self.switchOn = False
		self.restartGame = False
		self.tpcOn = False
		self.planeRotationMatrix = 0
		self.isUnusual = False
		
	def setGravityDirection(self, character_orientation):
		"""
		sets the direction of gravity by character_orientation
		
		@type character_orientation: Matrix
		@param character_orientation: the current orientation of the player character
		"""
		
		ori = character_orientation
				
		gravity_orientation = Matrix(ori[0], ori[1], ori[2]) * Vector([0, 0, -1])
		
		for i,k in enumerate(gravity_orientation):
			if fabs(k) <= 0.5:
				gravity_orientation[i] = 0
				self.gravityVector[i] = 0
			else:
				self.gravityAxis = i
				
				if k > 0:
					gravity_orientation[i] = 9.81
					self.gravityVector[i] = 1
				else:
					gravity_orientation[i] = -9.81
					self.gravityVector[i] = -1
				
				self.planeForceDirection = self.gravityVector[i]
		
		GameLogic.setGravity(gravity_orientation)
	
	def setFeedBack(self, controller, feedback_item, anim=False):
		"""
		sets the feedback pictures in the middle of the screen
		
		@type controller: the controler where the ipo actuator is connected to (only useful with anim=True)
		@param controller: SCA_PythonController
		@type feedback_item: String
		@param feedback_item: the name of the feedback that will appear
		@type anim: Boolean
		@param anim: True if there will be an animation (use skill items), False for not feasible actions and game options like sound
		"""
		
		scene = GameLogic.getCurrentScene()
		feedback = scene.objects["OBfeedback"]
		
		if feedback["show_feedback"]:
			for obj in feedback.children:
				obj.visible = False
				
		if not anim:
			no_action = scene.objects["OB" + feedback_item]
			no_action.visible = True
		else:
			feedback.children["OB" + feedback_item + "_activate"].visible = True
			anim = controller.actuators[feedback_item + "_activate"]
			controller.activate(anim)
		
		feedback["show_feedback"] = True
		feedback["timer"] = MY_SingleTimer().NOW
			

