from MyGameObjects import MY_GameObjectPassive
from MyScene import MY_SingleScene
from Mathutils import Vector, Matrix, RotationMatrix
from random import randint
from math import cos, sin, pi, ceil
from MyGameUtils import *


class MY_GamePlatform(MY_GameObjectPassive):
	"""
	class of the platform objects in the game for right rotation and translation etc.
	
	@ivar freeSidesPosition: a list element that consists of the connected side and the position where the new object can be connected to
	@type freeSidesPosition: List [Integer (0,1,2,3), Vector]; not very attractive :(
	@ivar origScale: the original scaling factor
	@type origScale: Vector
	@ivar midPoint: the center of the object (at least the right pivot, because at creation point of some elements the pivot is wrong. so you have to use this one to get the right rotations)
	@type midPoint: Vector
	"""
	
	def __init__(self, scene):
		"""
		@type scene: KX_Scene
		@param scene: the blender scene object where all other objects are connected to
		"""
		
		MY_GameObjectPassive.__init__(self, scene, "platform")
		
		if self.elementId >= 0:
			self.freeSidesPosition = []
			self.origScale = Vector(self.gObject.worldScale)
			self.gObject.localScale = [0, 0, 0]
			self.midPoint = Vector([0, 0, 0])
			myScene = MY_SingleScene()
			myScene.addObject(self)

	def setElement(self, side_location, prev_plane, rotated, direct=0):
		"""
		initiates the calculation of position, orientation, midpoint, boundingbox and worldplane
		
		@type side_location: List[Integer (0,1,2,3), Vector]
		@param side_location: to know exactly where the connection point is; that means what direction and at what position
		@type prev_plane: Integer (0,1,2)
		@param prev_plane: you have to know the previous plane (the plane of the object that this object is connected to) because of ninety degree turns
		@type rotated: Integer (0,1,2)
		@param rotated: whether th object is rotated (0=up, 1=down) or it is just connected in the same plane (2=normal)
		@type direct: Boolean
		@param direct: True if the object shall be spawned even ther is no other object to connect to (for example at the beginning)
		"""
		
		connected_dir = side_location[0]
		location = side_location[1]

		self.isRotated = rotated

		self.setLocation(connected_dir, location, prev_plane, direct)
		self.setOrientation(connected_dir, location, prev_plane)
		
		if self.isRotated != 2:
			self.setNewPlaneOnRotation(prev_plane, connected_dir)
		else:
			self.worldPlane = prev_plane
		
		self.midPoint = self.getPosition()
		self.boundingBox = self.getBoundingBox()

	def setLocation(self, connected_dir, location, prev_plane, direct=0):
		"""
		sets the location by use of the MyGameUtils interface
		
		@type connected_dir: Integer (0,1,2,3)
		@param connected_dir: one of the four directions an object can be connected to
		@type prev_plane: Integer (0,1,2)
		@param prev_plane: you have to know the previous plane (the plane of the object that this object is connected to) because of ninety degree turns
		@type location: Vector
		@param location: the position where the object shall be spawned
		@type direct: Boolean
		@param direct: True if the object shall be spawned even ther is no other object to connect to (for example at the beginning)
		"""
		
		half_width = self.gObject["width"] / 2
		half_height = self.gObject["height"] / 2

		self.gObject.worldPosition = location
		
		if self.isRotated != 2:
			self.gObject.worldPosition = GetPlatfVertTransDelta(location, connected_dir, prev_plane, half_width, half_height, self.isRotated)
		else:
			self.gObject.worldPosition = GetPlatfNormTransDelta(location, connected_dir, prev_plane, half_width, direct)

	def setOrientation(self, connected_dir, location, prev_plane):
		"""
		sets the orientation by use of the MyGameUtils interface
		
		@type connected_dir: Integer (0,1,2,3)
		@param connected_dir: one of the four directions an object can be connected to
		@type prev_plane: Integer (0,1,2)
		@param prev_plane: you have to know the previous plane (the plane of the object that this object is connected to) because of ninety degree turns
		@type location: Vector
		@param location: the position where the object shall be spawned
		"""
		
		if self.isRotated != 2:
			self.gObject.localOrientation = GetVertRotation(connected_dir, prev_plane)
		else:
			self.gObject.localOrientation = GetNormRotation(connected_dir, prev_plane)

	def getFreeSidesPositions(self):
		"""
		gets the positions of the not already connected sides of the object
		
		@rtype: List[Integer (0,1,2,3), Vector]
		@return: returns a list of all the free sides (direction and position)
		"""
		
		half_width = self.gObject["width"] / 2
		half_depth = self.gObject["depth"] / 2
		
		free_sides = []

		if self.connectedSides[0] == None:
			north = Vector([0, half_depth, 0])	
			free_sides.append([0, north])
		if self.connectedSides[1] == None:
			east = Vector([half_width, 0, 0])
			free_sides.append([1, east])
		if self.connectedSides[2] == None:
			south = Vector([0, -half_depth, 0])
			free_sides.append([2, south])
		if self.connectedSides[3] == None:
			west = Vector([-half_width, 0, 0])
			free_sides.append([3, west])
		
		if self.worldPlane == 0:
			for vector in free_sides:
				t_vector = vector[1]
				vector[1] = Vector(self.gObject.worldPosition) + t_vector
		elif self.worldPlane == 1:
			xrot_matrix = RotationMatrix(-90, 3, "x")
			for vector in free_sides:
				t_vector = xrot_matrix * Vector(vector[1])
				vector[1] = Vector(self.gObject.worldPosition) + t_vector
		elif self.worldPlane == 2:
			yrot_matrix = RotationMatrix(90, 3, "y")
			zrot_matrix = RotationMatrix(90, 3, "z")
			for vector in free_sides:
				t_vector = yrot_matrix * zrot_matrix * Vector(vector[1])
				vector[1] = Vector(self.gObject.worldPosition) + t_vector
		
		return free_sides # direction and position

	def getBoundingBox(self):
		"""
		gets the object's bounding box (the cubic hull of the object)
		
		@rtype: List
		@return: returns a list of vector points that represents the bounding box
		"""
		
		pos_x = self.gObject.worldPosition[0]
		pos_y = self.gObject.worldPosition[1]
		pos_z = self.gObject.worldPosition[2]
		
		half_width = self.gObject["width"] / 2
		half_depth = self.gObject["depth"] / 2
		half_height = self.gObject["height"] / 2

		bounding_box = []
		bounding_box.append(Vector([pos_x - half_width, pos_y - half_depth, pos_z - half_height]))
		bounding_box.append(Vector([pos_x - half_width, pos_y - half_depth, pos_z + half_height]))
		bounding_box.append(Vector([pos_x - half_width, pos_y + half_depth, pos_z - half_height]))
		bounding_box.append(Vector([pos_x - half_width, pos_y + half_depth, pos_z + half_height]))
		bounding_box.append(Vector([pos_x + half_width, pos_y - half_depth, pos_z - half_height]))
		bounding_box.append(Vector([pos_x + half_width, pos_y - half_depth, pos_z + half_height]))
		bounding_box.append(Vector([pos_x + half_width, pos_y + half_depth, pos_z - half_height]))
		bounding_box.append(Vector([pos_x + half_width, pos_y + half_depth, pos_z + half_height]))		

		if self.worldPlane == 1:
			xrot_matrix = RotationMatrix(90, 3, "x")
			for point in bounding_box:
				new_point = xrot_matrix * point
				point = new_point
		elif self.worldPlane == 2:
			yrot_matrix = RotationMatrix(90, 3, "y")
			for point in bounding_box:
				new_point = yrot_matrix * point
				point = new_point

		return bounding_box


class MY_GameStair(MY_GameObjectPassive):
	"""
	class of all stair-like objects to set the right location and orientation etc.
	
	@ivar isUp: True if the stair object goes upwards (will be initialized by a random for the lazy ones)
	@type isUp: Boolean
	@ivar origScale: the original scaling factor
	@type origScale: Vector
	@ivar midPoint: the center of the object (at least the right pivot, because at creation point of some elements the pivot is wrong. so you have to use this one to get the right rotations)
	@type midPoint: Vector
	"""
	
	def __init__(self, scene, name, scale):
		"""
		@type scene: KX_Scene
		@param scene: the blender scene object where all other objects are connected to
		@type name: String
		@param name: the name the object shall have in MY_SingleScene
		@type scale: Integer
		@param scale: the new scaling factor to increase the variance of the stair objects
		"""
		
		MY_GameObjectPassive.__init__(self, scene, name)
		
		if self.elementId >= 0:
			self.isUp = randint(0, 1)
			self.origScale = Vector([self.gObject.worldScale[0], self.gObject.worldScale[1] * scale, self.gObject.worldScale[2]])
			self.gObject.localScale = [0, 0, 0]
			self.midPoint = Vector([0, 0, 0])
			myScene = MY_SingleScene()
			myScene.addObject(self)

	def setElement(self, side_location, prev_plane):
		"""
		initiates the calculation of position, orientation, midpoint, boundingbox and worldplane
		
		@type side_location: List[Integer (0,1,2,3), Vector]
		@param side_location: to know exactly where the connection point is; that means what direction and at what position
		@type prev_plane: Integer (0,1,2)
		@param prev_plane: you have to know the previous plane (the plane of the object that this object is connected to) because of ninety degree turns
		"""
		
		connected_dir = side_location[0]
		location = side_location[1]
		
		self.worldDirection = connected_dir
		self.worldPlane = prev_plane

		self.gObject.worldPosition = location
		self.gObject.localOrientation = GetNormRotation(connected_dir, prev_plane, self.isUp)
		self.setMidPoint()
		self.boundingBox = self.getBoundingBox()
	
	def getFreeSidesPositions(self):
		"""
		gets the positions of the not already connected sides of the object
		
		@rtype: List[Integer (0,1,2,3), Vector]
		@return: returns the free sides (direction and position)
		"""
		
		o_dir = self.worldDirection
		
		north = self.connectedSides[0]
		east = self.connectedSides[1]
		south = self.connectedSides[2]
		west = self.connectedSides[3]
		
		if self.isUp:
			d_height = (self.gObject["height"] - 2.0) / 2.0
		else:
			d_height = ((self.gObject["height"] - 2.0) * -1) / 2.0
		
		y_scaling = self.origScale[1]
		d_depth = (self.gObject["depth"]/ 2.0) * y_scaling
		
		free_sides = []
		
		front_point = Vector([0, d_depth, d_height])
		back_point = Vector([0, -d_depth, -d_height])
		if o_dir == 0:
			if north == None:
				free_sides.append([0, front_point])
			if south == None:
				free_sides.append([2, back_point])
		elif o_dir == 1:
			if east == None:
				free_sides.append([1, front_point])
			if west == None:
				free_sides.append([3, back_point])
		elif o_dir == 2:
			if north == None:
				free_sides.append([0, back_point])
			if south == None:
				free_sides.append([2, front_point])
		elif o_dir == 3:
			if east == None:
				free_sides.append([1, back_point])
			if west == None:
				free_sides.append([3, front_point])
		
		for vector in free_sides:
			vector[1] = self.getRightPlanePoint(vector[1])

		return free_sides # direction and position

	def getRightPlanePoint(self, point):	
		"""
		gets the right point for a connection after all the creepy rotations of stairs and single steps :)
		
		@type point: Vector
		@param point: the point that is calculated in the xy-plane before any rotation and translation of the object
		
		@rtype: Vector
		@return: returns the right free side point
		"""
		
		zangle = 0
		if self.worldDirection == 1:
			zangle = 90
		elif self.worldDirection == 2:
			zangle = 180
		elif self.worldDirection == 3:
			zangle = -90

		point = RotationMatrix(zangle, 3, "z") * point

		rot_matrix = RotationMatrix(0, 3, "z")
		if self.worldPlane == 1:
			if self.worldDirection == 0 or self.worldDirection == 2:
				zrot_matrix = RotationMatrix(180, 3, "z")
				xrot_matrix = RotationMatrix(90, 3, "x")
				rot_matrix = xrot_matrix * zrot_matrix
			else:
				rot_matrix = RotationMatrix(90, 3, "x")
		elif self.worldPlane == 2:
			if self.worldDirection == 0 or self.worldDirection == 2:
				crank_rot = RotationMatrix(180, 3, "x")
			else:
				crank_rot = RotationMatrix(0, 3, "x")
			yrot_matrix = RotationMatrix(-90, 3, "y")
			zrot_matrix = RotationMatrix(90, 3, "z")
			rot_matrix = crank_rot * yrot_matrix * zrot_matrix
		
		point = rot_matrix * point

		point += Vector(self.midPoint)
			
		return point

	def getBoundingBox(self):
		"""
		gets the bounding box
		
		@rtype: List
		@return: returns a list of all the vector points that represents the bounding box
		"""
		
		sign = 1
		if not self.isUp:
			sign = -1
		
		depth_scaling = self.origScale[1]
		depth = (self.gObject["depth"] / 2.0) * depth_scaling
		
		half_width = self.gObject["width"] / 2
		
		height = sign * (self.gObject["height"]/ 2.0)

		t_bounding_box = []
		t_bounding_box.append(Vector([-half_width, -depth, -height]))
		t_bounding_box.append(Vector([-half_width, -depth, height]))
		t_bounding_box.append(Vector([-half_width, depth, -height]))
		t_bounding_box.append(Vector([-half_width, depth, height]))
		t_bounding_box.append(Vector([+half_width, -depth, -height]))
		t_bounding_box.append(Vector([+half_width, -depth, height]))
		t_bounding_box.append(Vector([+half_width, depth, -height]))
		t_bounding_box.append(Vector([+half_width, depth, height]))

		bounding_box = []

		for point in t_bounding_box:
			new_point = self.getRightPlanePoint(point)
			bounding_box.append(new_point)
		
		return bounding_box
	
	def setMidPoint(self):
		"""
		sets th mid point for rotation (stair objects rotation point is at the first face!)
		"""
		
		sign = 1
		if not self.isUp:
			sign = -1
		
		point = Vector([0, self.gObject["depth"] * 0.5 * self.origScale[1], sign * ((self.gObject["height"] * 0.5) - self.gObject["step_height"] * 0.5)])
		
		self.midPoint = self.getRightPlanePoint(point)
		self.midPoint += self.getPosition()			
			
		
