from MySpecialLists import MY_SingleIdList, MY_SingleElementList
from GameTypes import KX_GameObject
from MyScene import MY_SingleScene
from random import randint
from Mathutils import RotationMatrix, Matrix, Vector
from MyGameTimer import MY_SingleTimer
from MyGameWorld import MY_SingleGameWorld
from math import sin, cos, pi, fabs, floor


class MY_GameObjectActive():
	"""
	base class of all the objects that are dynamic and NOT temporary
	
	@ivar gObject: a blender game object
	@type gObject: KX_GameObject
	@ivar name: the name that represents the object in MY_SingleScene
	@type name: String
	@ivar type: a name which identifies the object as a specific class in MY_SingleScene
	@type type: String
	"""
	
	def __init__(self, scene, element):
		"""
		@type scene: KX_Scene
		@param scene: the blender scene object which holds all objects
		@type element: String
		@param element: the name of the light object which shall included into the MY_SingleScene class
		"""
		
		self.gObject = scene.objects["OB" + element]
		self.name = element
		self.type = "objectActive"

	def setLocalOrientation(self, orientation):
		"""
		sets the local orientation
		
		@type orientation: Matrix
		@param orientation: the new local orientation matrix
		"""
		
		self.gObject.localOrientation = orientation

	def getWorldOrientation(self):
		"""
		gets the world orientation
		
		@rtype: Matrix
		@return: a 3x3 matrix of the current orientation
		"""

		return self.gObject.worldOrientation

	def getPosition(self):
		"""
		gets the world position
		
		@rtype: Vector
		@return: a vector of the current world position
		"""
		
		return Vector(self.gObject.worldPosition)
		
	def setPosition(self, position):
		"""
		sets the world position
		
		@type position: Vector
		@param position: the new world position of the object
		"""
		
		self.gObject.worldPosition = position

	def getLinearVelocity(self, local):
		"""
		if 'local = True' it gets the local linear velocity
		otherwise it gets the global linear velocity
		
		@type local: Boolean
		@param local: True if you want the local linear velocity, otherwise it is global
		
		@rtype: List [vx, vy, vz]
		@return: the current linear velocity in all three directions
		"""
		
		return self.gObject.getLinearVelocity(local)
	
	def setLinearVelocity(self, speed, local=0):
		"""
		if 'local = True' it sets the local linear velocity
		otherwise it sets the global linear velocity
		
		@type speed: Vector
		@param speed: the max speed for all three directions
		@type local: Boolean
		@param local: True if the local velocity shall be set, otherwise global
		"""
		
		self.gObject.setLinearVelocity(speed, local)

	def deleteObject(self):
		"""
		deletes object and removes it from the scene
		"""
		
		self.gObject.endObject()
		myScene = MY_SingleScene()
		myScene.removeObject(self)


class MY_GameObjectActiveHidden():
	"""
	base class of all the objects that are dynamic but have a lifetime, so they are temporary ingame
	
	@ivar gObject: a blender game object
	@type gObject: KX_GameObject
	@ivar name: the name that represents the object in MY_SingleScene
	@type name: String
	@ivar type: a name which identifies the object as a specific class in MY_SingleScene
	@type type: String
	@ivar elementId: the unique identifier for this object
	@type elementId: Integer
	@ivar worldPlane: the number of the its current plane
	@type worldPlane: Integer (0,1,2)
	@ivar lifeTime: specifies the lifetime of the spawned item (negative = infinite)
	@type lifeTime: Integer
	"""
	
	def __init__(self, scene, element, spawn_point = 0):
		"""
		@type scene: KX_Scene
		@param scene: the blender scene object which holds all objects
		@type element: the name of the object
		@param element: String
		@type spawn_point: KX_GameObject
		@param spawn_point: the object where the object shall be spawned
		"""
		
		if spawn_point == 0:
			__center = scene.objects["OBcenter"]
			spawn_point = __center

		__freeId = MY_SingleIdList()

		self.gObject = scene.addObject(scene.objectsInactive["OB" + element], spawn_point)
		self.name = element
		self.type = "objectActiveTemp"
		self.elementId = __freeId.getElementId()
		self.gObject["id"] = self.elementId
		self.worldPlane = 0
		self.lifeTime = -1

	def getAlive(self, prota_pos):
		"""
		checks if object's lifetime is expired or object is out of game view
		
		@rtype: Boolean
		@return: True if the item is still alive, otherwise False
		"""
		
		result = True
	
		if self.lifeTime == -1:
			pass		
		elif floor(MY_SingleTimer().NOW) > floor(MY_SingleTimer().AGO) and (self.lifeTime-1) != -1:
			self.lifeTime -= 1
		
		distance_vector = self.getPosition() - prota_pos
		
		if self.lifeTime == 0 or distance_vector.length > 200:
			result = False

		return result

	def getPosition(self):
		"""
		gets the world position
		
		@rtype: Vector
		@return:  the current world position of the object
		"""
		
		return Vector(self.gObject.worldPosition)
	
	def setLinearVelocity(self, speed, local=True):
		"""
		just a forwarding of the blender setLinearVelocity
		
		@type speed: Vector
		@param speed: a vector with the speed of every single direction
		@type local: Boolean
		@param local: True if velocity has effect just locally, else globally
		"""
		
		self.gObject.setLinearVelocity(speed, local)
	
	def deleteObject(self):
		"""
		deletes object, removes it from the scene and frees its id
		"""
		
		self.gObject.endObject()
		freeId = MY_SingleIdList()
		freeId.setElementId(self.elementId)
		myScene = MY_SingleScene()
		myScene.removeObject(self)


class MY_GameObjectPassive():
	"""
	base class of all the objects in the game that are part of the environment, like platforms and stairs
	
	@ivar gObject: a blender game object
	@type gObject: KX_GameObject
	@ivar name: the name of the object
	@type name: String
	@ivar type: a name which identifies the object as a specific class in MY_SingleScene
	@type type: String
	@ivar elementId: a unique identifier
	@type elementId: Integer
	@ivar connectedSides: a dictionary (in form of a quadrupel) which contains the connected objects (otherwise: None) in specific way {north=+y, east=+x, south=-y, west=-x}
	@type connectedSides: Dict
	@ivar rootConnectObject: the connected object that leads to the players ground object
	@type rootConnectObject: MY_GameObject
	@ivar isRotated: three states: 0 = up; 1 = down; 2 = normal (in the xy-plane)
	@type isRotated: Integer (0,1,2)
	@ivar worldPlane: the objects plane information (0=xy; 1=xz; 2=yz)
	@type worldPlane: Integer (0,1,2)
	@ivar worldDirection: the global adjustment of the object (plane 0: 0=+y; 1=+x; 2=-y; 3=+x / plane 1: 0=+z; 1=+x; 2=-z; 3=+x / plane 2: 0=+z; 1=+y; 2=-z; 3=+y)
	@type worldDirection: Integer (0,1,2,3)
	@ivar popStart: the start time of popping
	@type popStart: Float
	@ivar popSteps: the number of steps for popping
	@type popSteps: Integer
	@ivar curPopStep: the current pop step
	@type curPopStep: Integer
	@ivar origScale: the original scale factor
	@type origScale: Vector
	@ivar curScale: the current scale factor
	@type curScale: Vector
	@ivar popDuration: the duration of popping
	@type popDuration: Integer
	@ivar deleteDelay: the amount of the delay for deleting the object; set to '<0' for no deletion
	@type deleteDelay: Integer
	@ivar boundingBox: contains all points of the bounding box which delimitates the object
	@type boundingBox: List
	@ivar hasItem: True if the object has an item to administrate while deletion, otherwise False
	@type hasItem: Boolean
	@ivar item: the item object that is on top/bottom of tis object
	@type item: MY_GameObject
	@ivar alive: True if object is destined for deletion (in the future)
	@type alive: Boolean
	@ivar hurdleObject: the hurdle object that is on THIS object (for oneighty turns), if no hurdle is near this variable is 'None'
	@type hurdleObject: MY_GameObject
	"""
	
	def __init__(self, scene, element):
		"""
		@type scene: KX_Scene
		@param scene: the blender scene object which holds all objects
		@type element: String
		@param element: the name of the object
		"""
		
		self.type = "objectPassive"
		self.elementId = -1
		self.gObject = 0
		self.name = element
		self.connectedSides = {0:None, 1:None, 2:None, 3:None} #[-1, -1, -1, -1] # list with the id's of connected objects
		self.rootConnectObject = None
		self.isRotated = 2
		self.worldPlane = 0
		self.worldDirection = 0 
		self.popStart = 0
		self.popSteps = 0 
		self.curPopStep = 0
		self.origScale = Vector([1, 1, 1])
		self.curScale = Vector([1, 1, 1])
		self.popDuration = 0
		self.deleteDelay = -1
		self.boundingBox = []
		self.hasItem = False
		self.item = None
		self.alive = True
		self.hurdleObject = None

		__obj_found = False
		__freeId = MY_SingleIdList()
		
		__center = scene.objects["OBcenter"]

		if element == "platform":
			for obj in scene.objectsInactive:
				if obj.name[2:10] == element:
					self.gObject = scene.addObject(obj, __center)
					__obj_found =True
		elif element == "single_step":
			for obj in scene.objectsInactive:
				if obj.name[2:13] == element:
					self.gObject = scene.addObject(obj, __center)
					__obj_found =True
		elif element == "stair":
			myStairList = MY_SingleElementList()
			choice = randint(0, len(myStairList.stairList)-1)
			if len(myStairList.stairList) > 0:
				obj = scene.objectsInactive[myStairList.stairList[choice]]
				self.gObject = scene.addObject(obj, __center)
			__obj_found =True
		else:
			print "no game element catched"

		if __obj_found and __freeId:
			self.elementId = __freeId.getElementId()
			self.gObject["id"] = self.elementId
			
	def getPlane(self):
		"""
		gets the world plane the object is in
		
		@rtype:  Integer (0,1,2)
		@return:
		 - returns '0' for xy plane
		 - returns '1' for xz plane
		 - returns '2' for yz plane
		"""
		
		return self.worldPlane
	
	def setNewPlaneOnRotation(self, prev_plane, connected_dir):
		"""
		sets new plane after orthogonal rotation dependent on the plane of connected object and its connected direction
		
		@type prev_plane: Integer (0,1,2)
		@param prev_plane: the plane of the connected object after ninety degree turn (dont know how to explain)
		@type connected_dir: Integer (0,1,2,3)
		@param connected_dir: on which side is the connection (0:+y, 1:+x, 2:-y, 3:-x)
		"""
		
		new_plane = 0
		if prev_plane == 0:
			if connected_dir == 0 or connected_dir == 2:
				new_plane = 1
			else:
				new_plane = 2
		elif prev_plane == 1:
			if connected_dir == 0 or connected_dir == 2:
				new_plane = 0
			else:
				new_plane = 2
		elif prev_plane == 2:
			if connected_dir == 0 or connected_dir == 2:
				new_plane = 0
			else:
				new_plane = 1

		self.worldPlane = new_plane
	
	def getPosition(self):
		"""
		gets the world position
		
		@rtype: Vector
		@return: the vector of the current world position
		"""
		
		return Vector(self.gObject.worldPosition)

	def setConnectedSide(self, connected_side, connected_object):
		"""
		sets list for connected sides with the right id at the right position in the array
		
		@type connected_side: Integer (0,1,2,3)
		@param connected_side: on which side is the connection (0:+y, 1:+x, 2:-y, 3:-x)
		@type connected_object: MY_GameObject
		@param connected_object: the object which is connected to THIS object
		"""
		
		self.connectedSides[connected_side] = connected_object

	def delConnectedSide(self, connected_object):
		"""
		deletes entry out of the connected side list
		
		@type connected_object: MY_GameObject
		@param connected_object: the connected object that was deleted
		"""

		for idx, obj in self.connectedSides.iteritems():
			if obj == connected_object:
				self.connectedSides[idx] = None
	
	def initPop(self, pop_steps, pop_start, duration):
		"""
		initialize popping when object is spawned
		
		@type pop_steps: Integer (> 0)
		@param pop_steps: how many steps are needed to pop up/down the object
		@type pop_start: Float
		@param pop_start: start time of popping
		@type duration: Float
		@param duration: for how long shall the object pop
		"""
		
		self.popSteps = pop_steps
		self.popStart = pop_start
		self.popDuration = duration
		self.gObject.localScale = [0, 0, 0]
		self.curScale = self.gObject.localScale
		
	def pop(self, do_pop, natal=True):
		"""
		pops the object with its initialized parameters
		
		@type do_pop: Boolean
		@param do_pop: do the pop (True) or not (False)
		@type natal: Boolean
		@param natal: whether it is the objects birth (True) or dead (False)
		"""
		
		if do_pop:
			myTimer = MY_SingleTimer()
		
			step_duration = self.popDuration / (self.curPopStep + 1)
			perc = ((myTimer.NOW * 1000) - (self.popStart * 1000)) / step_duration
		
			num_pi = 2 * pi * perc
			if natal:
				scale = cos(num_pi) * self.origScale
			else:
				scale = sin(num_pi) * self.origScale
		
			if self.curPopStep != self.popSteps:
				if natal:
					if num_pi > pi/2 and num_pi < 3.0/2.0 * pi:
						scale *= -1
				else:
					if num_pi > pi:
						scale *= -1

				self.gObject.localScale = Vector(self.curScale) + scale
			else:
				self.popSteps = 0
			
			if num_pi >= 2* pi:
				self.curPopStep += 1
				self.popStart = myTimer.NOW
		else:
			self.gObject.localScale = self.origScale
	
	def getChildren(self):
		"""
		gets the direct children which are connected to the object
		
		@rtype:  List
		@return: the list of the objects children
		"""
		
		children_list = []

		for idx,obj in self.connectedSides.iteritems():
			if obj != None and obj != self.rootConnectObject:
				recDelObj = obj

				if not recDelObj.gObject.invalid:
					children_list.append(recDelObj)
		return children_list
	
	def setAlive(self, alive):
		"""
		sets the object and its children to dead if necessary or the object to alive
		
		@type alive: Boolean
		@param alive: True if the object will live further more
		"""
		
		if not alive:
			if self.alive:

				self.deleteDelay = MY_SingleGameWorld().deleteDelay
				self.alive = False
		
				for i in self.getChildren():
					i.setAlive(False)
		else:
			self.deleteDelay = -1
			self.alive = True
	
	def deleteObject(self, now=False, recursive_delete=True):
		"""
		deletes the object. if a delay is given, the object will be deleted after the delay. with 'now' deletion can be forced
		all children can be deleted by setting the recursive_delete variable
		
		@type now: Boolean
		@param now: False if the deletion will be delayed, otherwise delete it NOW
		@type recursive_delete: Boolean
		@param recursive_delete: True if children will also been deleted
		
		@rtype: Boolean
		@return: returns 'True' if object was deleted, otherwise 'False'
		"""

		if self.deleteDelay > 0:
			myTimer = MY_SingleTimer()
			if floor(myTimer.NOW) > floor(myTimer.AGO):
				self.deleteDelay -= 1

		if self.deleteDelay == 0 or now:
			self.gObject.endObject()
			freeId = MY_SingleIdList()
			freeId.setElementId(self.elementId)
			myScene = MY_SingleScene()
			
			if self.name == "platform" and self.hasItem != 0:
				self.item.deleteObject()
				self.hasItem = 0
			
			for idx, obj in self.connectedSides.iteritems():
				if obj != None:
					freedObject = obj

					if not freedObject.gObject.invalid:
						freedObject.delConnectedSide(self)
			
			if recursive_delete:
				for j in self.getChildren():
					j.deleteObject(True)
				
			myScene.removeObject(self)
			
			return True
		else:
			return False


class MY_GameCamera():
	"""
	class for all the cameras used in the game
	
	@ivar gObject: a blender game object
	@type gObject: KX_GameObject
	@ivar name: the name that represents the object in MY_SingleScene
	@type name: String
	@ivar type: a name which identifies the object as a specific class in MY_SingleScene
	@type type: String
	"""
	
	def __init__(self, scene, element):
		"""
		@type scene: KX_Scene
		@param scene: the blender scene object which holds all objects
		@type element: String
		@param element: the name of the object
		"""
		
		self.gObject = scene.cameras["OB" + element]
		self.type = "objectCamera"
		self.name = element

		if self.gObject:
			myScene = MY_SingleScene()
			myScene.addObject(self)

	def pointInsideFrustum(self, point):
		"""
		checks if a given point is inside the cameras frustum
		
		@type point: Vector
		@param point: the point that will be checked if it is inside the camera frustum
		
		@rtype: String
		@return:
		 - returns OUTSIDE, if point is outside
		 - returns INSIDE, if point is inside
		"""
		
		return self.gObject.pointInsideFrustum(point)

	def boxInsideFrustum(self, box):
		"""
		checks if a given box is inside/intersect/outside the cameras frustum
		
		@type box: List of Vectors
		@param box: the box parameters that will be checked
		
		@rtype: Integer
		@return:
		 - returns 0, if box is outside
		 - returns 2, if box intersects
		 - returns 1, if box is inside
		"""
		
		if self.gObject.boxInsideFrustum(box) == self.gObject.OUTSIDE:
			return 0
		elif self.gObject.boxInsideFrustum(box) == self.gObject.INSIDE:
			return 1
		elif self.gObject.boxInsideFrustum(box) == self.gObject.INTERSECT:
			return 2

	def setLocalOrientation(self, orientation):
		"""
		sets the local orientation
		
		@type orientation: Matrix
		@param orientation: the new 3x3 orientation matrix
		"""
		
		self.gObject.localOrientation = orientation
	
	def getPosition(self):
		"""
		gets the local position
		
		@rtype: Vector
		@return: the vector of the current local position
		"""
		
		return Vector(self.gObject.localPosition)
	
	def deleteObject(self):
		"""
		deletes the object and removes it from the scene
		"""
		
		self.gObject.endObject()
		myScene = MY_SingleScene()
		myScene.removeObject(self)


class MY_GameLight():
	"""
	base class of all the light objects in the game
	
	@ivar gObject: a blender game object
	@type gObject: KX_GameObject
	@ivar name: the name that represents the object in MY_SingleScene
	@type name: String
	@ivar type: a name which identifies the object as a specific class in MY_SingleScene
	@type type: String
	"""
	
	def __init__(self, scene, element, name):
		"""
		@type scene: KX_Scene
		@param scene: the blender scene object which holds all objects
		@type element: String
		@param element: the name of the object
		@type name: String
		@param name: the name that represents the object in MY_SingleScene
		"""
		
		__center = scene.objects["OBcenter"]
		self.gObject = scene.addObject(scene.lights["OB" + element], __center)
		self.name = name
		self.type = "objectLight"
		
	def setEnergy(self, energy):
		"""
		sets the energy of the lamp
		
		@type energy: Float
		@param energy: the energy of the light object 
		"""
		
		self.gObject.energy = energy
		
	def setOrientation(self, angle, axis):
		"""
		sets the local orientation
		
		@type angle: Float
		@param angle: the new rotation angle
		@type axis: String (x,y,z)
		@param axis: the rotation axis
		"""
		
		self.gObject.localOrientation = RotationMatrix(angle, 3, axis)

	def setParent(self, parent_object):
		"""
		sets objects as children of given parent object
		
		@type parent_object: MY_GameObject
		@param parent_object:
		"""
		
		self.gObject.worldPosition = parent_object.gObject.worldPosition
		self.gObject.setParent(parent_object.gObject, 0, 1)
		
	def getPosition(self, local=1):
		"""
		gets the local position, if 'local' is True, otherwise it gets the global position
		
		@type local: Boolean
		@param local: True if you want to get the local position of the object, otherwise global
		
		@rtype: Vector
		@return: the vector of the current global/local position
		"""
		
		if local:
			position = Vector(self.gObject.localPosition)
		else:
			position = Vector(self.gObject.worldPosition)
	
		return position

	def deleteObject(self):
		"""
		deletes the object and removes it from the scene
		"""
		
		self.gObject.endObject()
		myScene = MY_SingleScene()
		myScene.removeObject(self)


