from MyGameObjects import MY_GameObjectActive, MY_GameObjectActiveHidden
from MyScene import MY_SingleScene
from math import sqrt, cos, sin, pi, fabs, floor
from time import clock
from Mathutils import RotationMatrix, Vector, Matrix
from MyGameTimer import MY_SingleTimer
from MySpecialLists import MY_InventoryList
from MyScene import MY_SingleScene
from MyGameWorld import MY_SingleGameWorld
from MyControlKeys import MY_SingleControlKeys
from MyConfigConnector import MY_SingleConfigConnector
import GameLogic

# character class	
class MY_GameCharacter(MY_GameObjectActive):
	"""
	class of the main game character, which the user has to play
	
	@ivar isJumping: True if player is jumping
	@type isJumping: Boolean
	@ivar isCrouching: True if player is crouching
	@type isCrouching: Boolean
	@ivar stepUp: True if payer wants to go upstairs
	@type stepUp: Boolean
	@ivar stepDown: Amount of the time to get the player back to ground
	@type stepDown: Integer
	@ivar zLook: amount of the rotation angle around the z-axis
	@type zLook: Float
	@ivar wallStartTime: the start time of a wallturn
	@type wallStartTime: Float
	@ivar wallTurn: True if player makes a ninety or oneighty turn
	@type wallTurn: Boolean
	@ivar wallTurnEnd: True if a turn just ended
	@type wallTurnEnd: Boolean
	@ivar turnDuration: the duration of a turn 
	@type turnDuration: Integer
	@ivar worldPlane: the players current plane
	@type worldPlane: Integer (0,1,2)
	@ivar oppositePlane: True if player is on one of the planes which gravity vector goes in th direction of z, -x or -y
	@type oppositePlane: Boolean
	@ivar globalRotationX: the rotation amount about x-axis when turning
	@type globalRotationX: Integer
	@ivar globalRotationY: the rotation amount about y-axis when turning
	@type globalRotationY: Integer
	@ivar zTurn: the rotation about z-axis; just for calculating the wallturn while turning
	@type zTurn: Float
	@ivar cosX: the cosinus amount of looking up and down
	@type cosX: Float
	@ivar sinX: the sinus amount of looking up and down
	@type sinX: Float
	@ivar angleXTotal: the global rotation about the x-axis of the player (player rotates finally just about x and y)
	@type angleXTotal: Integer
	@ivar angleYTotal: the global rotation about the y-axis of the player (player rotates finally just about x and y)
	@type angleYTotal: Integer
	@ivar angleX: just a temporary variable to leave the angleXTotal variable untouched
	@type angleX: Float
	@ivar angleY: just a temporary variable to leave the angleYTotal variable untouched
	@type angleY: Float
	@ivar groundObject: the object that is right under the player's feet
	@type groundObject: MY_GameObject
	@ivar inventoryList: a list that contains all amounts of your inventoriy items 
	@type inventoryList: MY_InventoryList
	@ivar shotTime: the time when you shot a bullet
	@type shotTime: Float
	@ivar shotInterval: the interval of shots
	@type shotInterval: Float
	@ivar flyTime: the time when you began to leave the ground
	@type flyTime: Float
	@ivar mag: the total amount of the player's mag
	@type mag: Integer
	@ivar ammo: the amount of bullets the player has in his gun
	@type ammo: Integer
	@ivar aim: True if player aims at a target, otherwise False
	@type aim: Boolean
	@ivar hasVisor: True if player uses visor, else False
	@type hasVisor: Boolean
	@ivar visorAnimStart: the time when player clicked the v-key to get his visor
	@type visorAnimStart: Float
	@ivar godMode: True if player is in godmode; normally after a hurdle hit to stay alive and not to lose all your health
	@type godMode: Boolean
	@ivar godStartTime: the time when the player godmode started
	@type godStartTime: Float
	@ivar rotationPointObject: the object to set at the center of a rotation object while doing a oneighty
	@type rotationPointObject: KX_GameObject
	@ivar visualInput: the object to set at the screen to let the player know what orientation he/she/it has
	@type visualInput: KX_GameObject
	@ivar visualInputObject: the actual object the payer will see as visual feedback for his/her orientation
	@type visualInputObject: KX_GameObject
	@ivar lookCamera: the camera object the player uses for looking around (for example: first person or third person)
	@type lookCamera: KX_Camera
	@ivar superGun: the player has collected the rare super gun in the game
	@type superGun: Boolean
	@ivar useSuperGun: True if the player is currently using the super gun, else False
	@type useSuperGun: Boolean
	"""
	
	def __init__(self, scene, element):
		MY_GameObjectActive.__init__(self, scene, element)
		self.isJumping = True
		self.isCrouching = False
		self.stepUp = False
		self.stepDown = 0
		self.zLook = 0.0
		self.wallStartTime = 0.0
		self.wallTurn = False
		self.wallTurnEnd = True
		self.turnDuration = 0
		self.worldPlane = 0
		self.oppositePlane = False
		self.globalRotationX = 0
		self.globalRotationY = 0
		self.zTurn = 0
		self.cosX = 1.0
		self.sinX = 0.0
		self.angleXTotal = 0
		self.angleX = 0
		self.angleYTotal = 0
		self.angleY = 0
		self.groundObject = None
		self.inventoryList = MY_InventoryList()
		self.shotTime = 0
		self.shotInterval = 0
		self.flyTime = 0
		self.mag = 0
		self.ammo = 0
		self.aim = False
		self.hasVisor = False
		self.visorAnimStart = 0.0
		self.godMode = False
		self.godStartTime = 0.0
		self.rotationPointObject = None
		self.visualInput = scene.objects["OBvisual_input"]
		self.visualInputObject = scene.objects["OBvisual_input_object"]
		self.lookCamera = scene.objects["OBfpc"]
		self.superGun = False
		self.useSuperGun = False
		
		self.inventoryList.fillWithConfigValues()
		
		for i in self.inventoryList.inventoryList:
			real_name = "OB" + i[1]
			gauge_icon = scene.objects[real_name]
			gauge_icon.visible = True
			itemGauge = MY_GameGauge(scene, i[1])
			itemGauge.setPosition(Vector(gauge_icon.worldPosition) - Vector([-0.007, 0, 0.045]))
			itemGauge.theText = i[0]
			itemGauge.setText()
			if i[1] == "sniper":
				if i[0] == 0:
					gauge_icon.visible = False
				else:
					self.superGun = True
			itemGauge.gObject.visible = gauge_icon.visible
		
		if self.gObject:
			myScene = MY_SingleScene()
			myScene.addObject(self)
	
	def setView(self, window_metric):
		"""
		sets the players view. to switch between first person and third persion view
		
		@type window_metric: List [Integer, Integer]
		@param window_metric: a list of thw width and height of the game window
		"""
		
		window_width = window_metric[0]
		window_height = window_metric[1]
		self.lookCamera.setViewport(0, 0, window_width, window_height)
		self.lookCamera.useViewport = True

	def jump(self, speed, force):
		"""
		sets the jump digit of the speed vector with the given force
		
		@type speed: Vector
		@param speed: the speed vector of the player
		@type force: Integer
		@param force: the power of the jump
		
		@rtype: Vector
		@return: the prepared speed vector
		"""

		self.isJumping = True
		speed[2] += force
		step_down_amount = 6
		
		return speed
	
	def marioJump(self, hurdle_pos):
		"""
		sets the mario jump, to get out of the way of an enemy after get hurt once
		
		@type hurdle_pos: Vector
		@param hurdle_pos: the global position of the hurdle
		"""
		
		force_up = 10
		jump = MY_SingleGameWorld().gravityVector * -force_up
		
		max_ground_speed = 10
		
		direction_vector = self.getPosition() - hurdle_pos
		ground_speed = []
		
		for i,k in enumerate(MY_SingleGameWorld().gravityVector):
			if fabs(k) < 1: # ground movement
				ground_speed.append([i, direction_vector[i]])
		
		full_ground_speed = fabs(ground_speed[0][1]) + fabs(ground_speed[1][1])
		sign = 1
		if ground_speed[0][1] < 0:
			sign = -1
		jump[ground_speed[0][0]] = fabs(ground_speed[0][1]) / full_ground_speed * sign * max_ground_speed
		sign = 1
		if ground_speed[1][1] < 0:
			sign = -1
		jump[ground_speed[1][0]] = fabs(ground_speed[1][1]) / full_ground_speed * sign * max_ground_speed

		self.gObject.setLinearVelocity(jump)
	
	def move(self, fwd, back, left, right, force):
		"""
		calculates the movement of the object, dependent on given input parameters (direction and force)
		
		@type fwd: Boolean
		@param fwd: the power of forward movement
		@type back: Boolean
		@param back: the power of backward movement
		@type left: Boolean
		@param left: the power of left movement
		@type right: Boolean
		@param right: the power of right movement
		@type force: Integer
		@param force: the speed of the player's total movement
		
		@rtype: Vector
		@return: the prepared speed vector for the player's character
		"""
		
		sqrt2over2 = sqrt(2)/2

		fspeed = force * (fwd - back)
		sspeed = force * (right - left)
	
		if fspeed and sspeed:
			fspeed *= sqrt2over2
			sspeed *= sqrt2over2
		
		speed = [sspeed, fspeed, 0]

		return speed

	def look(self, mouse_screen_pos, dtime, sensitivity):
		"""
		sets the players actual camera view dependent on user's mouse input, elapsed time and sensitivity
		
		@type mouse_screen_pos: List [Integer, Integer]
		@param mouse_screen_pos: the position of the mouse on the screen related to the window metric
		@type dtime: Float
		@param dtime: the delta of time between two logical ticks
		@type sensitivity: Float
		@param sensitivity: the speed of turning your eyes around
		"""
		
		if self.wallTurn:
			self.planeTurn()
		else:
			# Mouse sensitivity
			xsensitivity = -sensitivity
			ysensitivity = sensitivity

			## z axis rotation of character
			oz = self.zLook
			zrot = 180 * xsensitivity * dtime * mouse_screen_pos[0]
			self.zLook = oz + zrot
		
			# x axis rotation of cam
			osx = self.sinX
			ocx = self.cosX
			xrot = (pi * ysensitivity) * dtime * mouse_screen_pos[1]
			sxrot = sin(xrot)
			cxrot = cos(xrot)
			self.cosX = -osx * sxrot + ocx * cxrot
			self.sinX =  osx * cxrot + ocx * sxrot

			# prevent looking top or bottom
			if self.cosX < 0.5:
				self.cosX = ocx
				self.sinX = osx
		
			wall_rotx = RotationMatrix(self.angleXTotal, 3, "x")
			wall_roty = RotationMatrix(self.angleYTotal, 3, "y")
			wall_rotz = RotationMatrix(self.zLook, 3, "z")
			MY_SingleGameWorld().planeRotationMatrix = wall_rotx * wall_roty
			self.gObject.localOrientation = wall_rotx * wall_roty * wall_rotz
		
			self.lookCamera.localOrientation = Matrix( [1, 0, 0],
								[0, -self.sinX, -self.cosX],
								[0, self.cosX, -self.sinX] )
		
		if MY_SingleGameWorld().specialKeys:
			self.visualInputObject.setVisible(True, True)
			self.visualInputObject.worldPosition = self.visualInput.worldPosition
		else:
			self.visualInputObject.setVisible(False, True)
	
	def initOneighty(self, turn_duration):
		"""
		initialize the 'Oneighty' with the right parameters and given duration
		it sets the used member variables with the right values to know how to rotate and the local rotation of the player afterwards
		and parents the player to the rotating ground object
		
		@type turn_duration: Integer
		@param turn_duration: how fast/slow the turn shall be
		"""
		
		self.godMode = True
		self.wallStartTime = MY_SingleTimer().NOW * 1000
		self.wallTurn = True
		self.turnDuration = turn_duration
		
		print "world plane: ",
		print self.worldPlane
		print "angle x: ",
		print self.angleXTotal
		print "angle y: ",
		print self.angleYTotal
		print "opposite: ",
		print self.oppositePlane
		
		myGroundObject = self.groundObject

		direction = myGroundObject.worldDirection
		name = myGroundObject.name

		if self.worldPlane == 0:
			if direction == 0 or direction == 2:
				if name == "single_step":
					self.globalRotationY = 180
				elif name == "stair":
					self.globalRotationX = 180
			else:
				if name == "single_step":
					self.globalRotationX = 180
				elif name == "stair":
					self.globalRotationY = 180
		elif self.worldPlane == 1:
			if direction == 0 or direction == 2:
				if name == "single_step":
					self.zTurn = 180
				self.globalRotationX = 180
			else:
				if name == "stair":
					self.zTurn = 180
				self.globalRotationX = 180
		elif self.worldPlane == 2:
			if direction == 0 or direction == 2:
				if name == "single_step":
					self.zTurn = 180
				self.globalRotationY = 180
			else:
				if name == "stair":
					self.zTurn = 180
				self.globalRotationY = 180
		
		self.wallTurnEnd = False
		
		if self.oppositePlane:
			self.oppositePlane = False
		else:
			self.oppositePlane = True
		
		if self.globalRotationX == 180 and self.angleYTotal != 0:
			self.angleXTotal = self.angleYTotal
			self.angleYTotal = 0
		elif self.globalRotationY == 180 and self.angleXTotal != 0:
			self.angleYTotal = -self.angleXTotal
			self.angleXTotal = 0
		
		self.rotationPointObject.worldPosition = myGroundObject.midPoint
		if self.groundObject != None:
			self.groundObject.gObject.setParent(self.rotationPointObject, 0, 1)
		self.gObject.setParent(self.rotationPointObject, 0, 1)
		if self.groundObject.hurdleObject != None:
			self.groundObject.hurdleObject.gObject.setParent(self.rotationPointObject, 0, 1)
		self.gObject.setLinearVelocity([0, 0, 0])
		GameLogic.setGravity([0,0,0])
	
	def initWallRun(self, global_collision_side, hit_plane, turn_duration):
		"""
		initialize the wall turn. it sets the used member variables with the right values
		
		@type global_collision_side: Integer (0,1,2,3)
		@param global_collision_side: where is the wall globally? (at least to know rotation axis)
		@type hit_plane: Integer (0,1,2)
		@param hit_plane: the plane of the hit object, to know the player's world plane after rotation
		@type turn_duration: Integer
		@param turn_duration: time to rotate
		"""
		
		self.godMode = True
		self.wallStartTime = MY_SingleTimer().NOW * 1000
		self.wallTurn = True
		self.turnDuration = turn_duration
		
		print "world plane: ",
		print self.worldPlane
		print "angle x: ",
		print self.angleXTotal
		print "angle y: ",
		print self.angleYTotal
		print "opposite: ",
		print self.oppositePlane

		sign = 1
		if self.oppositePlane:
			sign = -1
		
		if self.worldPlane == 0:
			if global_collision_side == "front":
				self.globalRotationX = sign * -90
				self.oppositePlane = False
			elif global_collision_side == "right":
				self.globalRotationY = sign * 90
				self.oppositePlane = False
			elif global_collision_side == "back":
				self.globalRotationX = sign * 90
				self.oppositePlane = True
			elif global_collision_side == "left":
				self.globalRotationY = sign * -90
				self.oppositePlane = True
		elif self.worldPlane == 1:
			if global_collision_side == "front":
				self.globalRotationX = sign * -90
				self.oppositePlane = True
			elif global_collision_side == "right":
				self.globalRotationY = 90
				self.zTurn = sign * 90
				self.oppositePlane = False
			elif global_collision_side == "back":
				self.globalRotationX = sign * 90
				self.oppositePlane = False
			elif global_collision_side == "left":
				self.globalRotationY = -90
				self.zTurn = sign * -90
				self.oppositePlane = True
		elif self.worldPlane == 2:
			if global_collision_side == "front":
				self.globalRotationY = sign * 90
				self.oppositePlane = True
			elif global_collision_side == "right":
				self.globalRotationX = 90
				self.zTurn = sign * 90
				self.oppositePlane = True
			elif global_collision_side == "back":
				self.globalRotationY = sign * -90
				self.oppositePlane = False
			elif global_collision_side == "left":
				self.globalRotationX = -90
				self.zTurn = sign * -90
				self.oppositePlane = False
		
		if fabs(self.globalRotationX) == 90 and self.angleYTotal == -180:
			self.angleXTotal = self.angleYTotal * (self.globalRotationX/90)
			self.angleYTotal = 0
			self.zLook -= 180
		elif fabs(self.globalRotationX) == 90 and self.angleYTotal == 180:
			self.angleXTotal = -self.angleYTotal * (self.globalRotationX/90)
			self.angleYTotal = 0
			self.zLook -= 180
		elif fabs(self.globalRotationY) == 90 and self.angleXTotal == 180:
			self.angleYTotal = -self.angleXTotal * (self.globalRotationY/90)
			self.angleXTotal = 0
			self.zLook -= 180
		elif fabs(self.globalRotationY) == 90 and self.angleXTotal == -180:
			self.angleYTotal = self.angleXTotal * (self.globalRotationY/90)
			self.angleXTotal = 0
			self.zLook -= 180
		
		self.zLook %= 360
		
		self.wallTurnEnd = False
		self.worldPlane = hit_plane
		
		self.rotationPointObject.worldPosition = self.getPosition()
		self.gObject.setParent(self.rotationPointObject, 0, 1)
		self.gObject.setLinearVelocity([0, 0, 0])
		GameLogic.setGravity([0,0,0])
		
	def planeTurn(self):
		"""
		do the Oneighty
		"""
		
		timer_now = MY_SingleTimer().NOW * 1000
		
		wall_rot = RotationMatrix(0, 3, "x")
		
		if timer_now < self.wallStartTime + self.turnDuration and not self.wallTurnEnd:
			perc = (timer_now - self.wallStartTime) / self.turnDuration
			pangle = 0
			
			if self.zTurn != 0:
				pangle = self.zTurn * perc
				
				axis = "z"
			else:
				axis = "x"
				if self.globalRotationX != 0:
					pangle = self.globalRotationX * perc
				else:
					pangle = self.globalRotationY * perc
					axis = "y"

			wall_rot_vis = RotationMatrix(pangle, 3, axis)
				
		elif not self.wallTurnEnd:
			
			if self.zTurn != 0:
				pangle = self.zTurn
				axis = "z"
			else:
				axis = "x"
				if self.globalRotationX != 0:
					pangle = self.globalRotationX
				else:
					pangle = self.globalRotationY
					axis = "y"
			
			self.rotationPointObject.localOrientation = RotationMatrix(pangle, 3, axis)
			
			if self.globalRotationX != 0:
				self.angleXTotal += self.globalRotationX
			else:
				self.angleXTotal = 0
			
			if self.globalRotationY != 0:
				self.angleYTotal += self.globalRotationY
			else:
				self.angleYTotal = 0
			
			if self.angleXTotal < -180:
				self.angleXTotal += 360
			elif self.angleXTotal > 180:
				self.angleXTotal -= 360
			
			if self.angleYTotal < -180:
				self.angleYTotal += 360
			elif self.angleYTotal > 180:
				self.angleYTotal -= 360

			print ""
			print "world plane: ",
			print self.worldPlane
			print "angle x: ",
			print self.angleXTotal
			print "angle y: ",
			print self.angleYTotal
			print "opposite: ",
			print self.oppositePlane
			print "end_rot"
			print "------------------"

			wall_rotx = RotationMatrix(self.angleXTotal, 3, "x")
			wall_roty = RotationMatrix(self.angleYTotal, 3, "y")
		
			wall_rot = wall_roty * wall_rotx
						
			self.zLook += self.zTurn	
			self.zTurn = 0
			
			self.globalRotationX = 0
			self.globalRotationY = 0
			
			self.wallTurnEnd = True
		
			self.wallStartTime = 0
			self.wallTurn = False
			
			if self.groundObject != None:
				self.groundObject.gObject.removeParent()
			if self.groundObject.hurdleObject != None:
				self.groundObject.hurdleObject.gObject.removeParent()
			self.gObject.removeParent()
			
			wall_rot_vis = RotationMatrix(0, 3, "z")
			MY_SingleGameWorld().setGravityDirection(wall_rot)
			
			self.godMode = False
			
		self.rotationPointObject.localOrientation = wall_rot_vis
		
		self.gObject.setLinearVelocity([0,0,0])
		
		if MY_SingleGameWorld().specialKeys:
			MY_SingleControlKeys().setControlKeysEffect(self.worldPlane, self.oppositePlane)
	
	def setInventoryItem(self, item_name, add):
		"""
		sets or removes given item into the inventory
		
		@type item_name: String
		@param item_name: the name of the item that was used/collected
		@type add: Boolean
		@param add: True if the item was collected, False if it was used
		
		@rtype: Boolean
		@return: 'True' if item is in inventory, else 'False'
		"""
		if MY_SingleGameWorld().isUnusual:
			exist = True
		else:
			exist = self.inventoryList.useItem(item_name, add)
			self.setInventoryGauge()

		return exist
	
	def setInventoryGauge(self):
		"""
		refresh the inventory gauge on the screen and in the config parser
		"""
		if MY_SingleGameWorld().isUnusual:
			for i in self.inventoryList.inventoryList:
				if i[1] != "sniper":
					itemGauge = MY_SingleScene()[i[1]]
					itemGauge.gObject.visible = False
					MY_SingleScene().scene.objects["OB" + i[1] + "_inf"].visible = True
		else:
			for i in self.inventoryList.inventoryList:
				itemGauge = MY_SingleScene()[i[1]]
				MY_SingleScene().scene.objects["OB" + i[1] + "_inf"].visible = False
				itemGauge.gObject.visible= True
				itemGauge.theText = i[0]
				itemGauge.setText()
				if i[1] == "sniper":
					if i[0] > 0:
						GameLogic.getCurrentScene().objects["OB" + i[1]].visible = True
						self.superGun = True
					else:
						GameLogic.getCurrentScene().objects["OB" + i[1]].visible = False
						self.superGun = False
				itemGauge.gObject.visible = GameLogic.getCurrentScene().objects["OB" + i[1]].visible

	def getItemLifeTime(self, name):
		"""
		sets the lifetime of an used item
		
		@type name: String
		@param name: the name of the used item to get its lifetime
		
		@rtype: Boolean
		@return: 'True' if item is still alive, else 'False'
		"""
		
		item = self.inventoryList[name]
		if item[2] != item[3]:
			myTimer = MY_SingleTimer()
			if floor(myTimer.NOW) > floor(myTimer.AGO):
				if item[2] <= 0:
					item[2] = item[3]
				else:
					item[2] -= 1
					
			return True
		
		return False
	
	def setMerchItem(self, merch_name):
		"""
		adds a given merchandise item into the inventory
		
		@type merch_name: String
		@param merch_name: the name of the merch item that was collected
		"""
		
		self.inventoryList.addMerchItem(merch_name)
	
	def setWarp(self):
		"""
		do a space warp
		"""
		
		pass
		
	def reloadAmmo(self):
		"""
		reloads your ammo
		"""
		
		self.ammo = self.mag


# sky sphere class
class MY_SkySphere(MY_GameObjectActive):
	"""
	class of the unversum sphere which centers on the player
	"""

	def __init__(self, scene, element):
		"""
		@type scene: KX_Scene
		@param scene: the blender scene object which holds all objects
		@type element: String
		@param element: the name of the object which shall included into the MY_SingleScene class
		"""
		
		MY_GameObjectActive.__init__(self, scene, element)
		
		if self.gObject:
			myScene = MY_SingleScene()
			myScene.addObject(self)

	def setAction(self, axis, location):
		"""
		sets the sky dome rotation
		
		@type axis: Integer (0,1,2)
		@param axis: the rotation axis of the sky sphere
		@type location: Vector
		@param location: the new position of the sphere's centre
		"""
		
		rad = ((clock()*0.001) % 2) * pi

		cosa = cos(rad)
		sina = sin(rad)

		self.gObject.position = location
		self.gObject.orientation = [ [cosa, -sina, 0], [sina, cosa, 0], [0, 0, 1] ]


# gauge class	
class MY_GameGauge(MY_GameObjectActiveHidden):
	"""
	class of the gauge elements, like inventory, health etc.
	
	@ivar name: the object's name 
	@type name: String
	@ivar theText: the text the gauge shall display (Integer for gauge_num and String for gauge_text)
	@type theText: String or Integer
	"""
	
	def __init__(self, scene, element, text=False):
		"""
		@type scene: KX_Scene
		@param scene: the blender scene object which holds all objects
		@type text: Boolean
		@param text: True if it is a text otherwise it is a number
		"""
		
		if text:
			MY_GameObjectActiveHidden.__init__(self, scene, "gauge_text")
			self.gObject.setParent(scene.objects["OB" + element], 0, 0)
			self.gObject.orientation = RotationMatrix(-90, 3, "x")
		else:
			MY_GameObjectActiveHidden.__init__(self, scene, "gauge_num")
			self.gObject.setParent(scene.objects["OB" + element], 0, 0)
			self.gObject.localScale = [0.015, 0.015, 0.015]
			self.gObject.orientation = RotationMatrix(-90, 3, "x")
			
		self.name = element
		self.theText = 0
		
		if self.gObject:
			myScene = MY_SingleScene()
			myScene.addObject(self)

	def setText(self):
		"""
		sets the text to be displayed in the player's gauge
		"""
		
		self.gObject["Text"] = self.theText
		
	def setVisible(self, visible):
		"""
		sets the visibility of a gauge item
		
		@type visible: Boolean
		@param visible: sets the visibility of the text (True = visible; False = not visible)
		"""
		
		self.gObject.setVisible(visible, False)

	def setPosition(self, position):
		"""
		sets the world position
		
		@type position: Vector
		@param position: the new world position of the text
		"""
		
		self.gObject.worldPosition = position


