from MyGameObjects import MY_GameObjectActiveHidden, MY_GameLight
from MyScene import MY_SingleScene
from time import clock
import math
from math import pi, cos, sin, fabs, floor
from random import randint
from MyGameWorld import MY_SingleGameWorld
from MySpecialLists import MY_SingleElementList, MY_SingleIdList
from MyGameTimer import MY_SingleTimer
from Mathutils import RotationMatrix, Vector, AngleBetweenVecs, Matrix
from MyGameUtils import GetGlobalOrientationToMe
from GameTypes import KX_GameObject

# sky light class
class MY_SkyLight(MY_GameLight):
	"""
	class of the sky lights to simulate a sun movement
	
	@ivar fadeStart: the start time for fading lights (on/off)
	@type fadeStart: Float
	@ivar oldEnergy: the starting energy of the light object 
	@type oldEnergy: Float
	@ivar isFading: whether the light is fading or not
	@type isFading: Boolean
	"""
	
	def __init__(self, scene, element, name):
		"""
		@type scene: KX_Scene
		@param scene: the blender scene object which holds all objects
		@type element: String
		@param element: the name of the light object which shall included into the MY_SingleScene class
		@type name: String
		@param name: the name the object shall have in MY_SingleScene
		"""
		
		MY_GameLight.__init__(self, scene, element, name)
		
		self.fadeStart = 0
		self.oldEnergy = 0
		self.isFading = False
		myScene = MY_SingleScene()
		myScene.addObject(self)
	
	def setEnergyFade(self, max_amount, duration):
		"""
		fades the energy of the light object to a given amount in a given time
		
		@type max_amount: Float
		@param max_amount: amount to fade to
		@type duration: Float		
		@param duration: duration which shall be used to fade the light's energy
		"""
		
		cur_fade_time = MY_SingleTimer().NOW - self.fadeStart

		if (cur_fade_time <= duration):
			if not self.isFading:
				self.isFading = True
				self.oldEnergy = self.gObject.energy
			
			cur_amount = cur_fade_time / duration * (max_amount - self.oldEnergy)
			self.setEnergy(cur_amount)
		else:
			self.isFading = False
	
	def setRotation(self, plane, radius, pivot):
		"""
		set an infinty rotation for day night effect
		rotation axis depends on plane
		
		@type plane: Integer [0,1,2]
		@param plane: the plane on which the light shall rotate
		@type radius: Float
		@param radius: the radius for the rotation
		@type pivot: Vector
		@param pivot: the middle of the circle movement
		"""

		rad = ((clock() * 0.1) % 2) * pi
		cosinus = radius * cos(rad)
		sinus = radius * sin(rad)
		
		new_position = [0, 0, 0]

		if plane == "xy":
			new_position = Vector([cosinus, sinus, 0.0])
		elif plane == "xz":
			new_position = Vector([cosinus, 0.0, sinus])
		elif plane == "yz":
			new_position = Vector([0.0, cosinus, sinus])
		else:
			print "wrong axis"
		
		self.gObject.worldPosition = new_position + pivot

# item class
class MY_GameItem(MY_GameObjectActiveHidden):
	"""
	class of the game items that can be collected by the player and are positioned onm top/bottom of platforms
	
	@ivar rotationDuration: how long shall it take to make a full circle rotation
	@type rotationDuration: Float
	@ivar rotationStart: start time of the circle rotation
	@type rotationStart: Float
	@ivar positionRotationX: specifies the tilt of the item dependent on its ground plane
	@type positionRotationX: Float
	@ivar positionRotationY: specifies the tilt of the item dependent on its ground plane
	@type positionRotationY: Float
	@ivar parentObject: the items parent object to know which object can be relieved of the item
	@type parentObject: MY_GameObject
	@ivar worldPlane: holds the number of the world plane
	@type worldPlane: Integer (0,1,2)
	"""

	def __init__(self, scene, element_name, parent_object):
		"""
		@type scene: KX_Scene
		@param scene: the blender scene object which holds all objects
		@type element_name: String
		@param element_name: the name of the light object which shall included into the MY_SingleScene class
		@type parent_object: MY_GameObject
		@param parent_object: the game object the item shall parented to
		"""
		
		MY_GameObjectActiveHidden.__init__(self, scene, element_name)
		myItemList = MY_SingleElementList().itemList
		item_choice = randint(0, len(myItemList)-1)
		self.gObject.replaceMesh(scene.objectsInactive[myItemList[item_choice][0]].meshes[0])
		#self.gObject.replaceMesh(scene.objectsInactive["OBitem_sniper"].meshes[0])
		self.rotationDuration = 2.0
		self.rotationStart = 0
		self.positionRotationX = 0
		self.positionRotationY = 0
		self.parentObject = parent_object
		self.worldPlane = parent_object.worldPlane
		self.setPosition()
		self.setParentStatus(True, self)
		
		myScene = MY_SingleScene()
		myScene.addObject(self)
	
	def getPosition(self):
		"""
		get the world position of the item
		
		@rtype: Vector
		@return: the vector of the current world position
		"""
		
		return Vector(self.gObject.worldPosition)
		
	def setPosition(self):
		"""
		set an item at top or bottom of a parented platform
		"""
		
		top = randint(0, 1)
		sign = 1
		if top:
			sign = -1
		
		# sets position adjustment dependent on the plane
		if self.worldPlane == 0:
			self.gObject.worldPosition = self.parentObject.getPosition() + (-2 * Vector([0, 0, sign * 1]))
		elif self.worldPlane == 1:
			self.gObject.worldPosition = self.parentObject.getPosition() + (-2 * Vector([0, sign * 1, 0]))
		elif self.worldPlane == 2:
			self.gObject.worldPosition = self.parentObject.getPosition() + (-2 * Vector([sign * 1, 0, 0]))
		
		# sets the right rotation dependent on the actual orientation to the parent platform
		side = GetGlobalOrientationToMe(self, self.parentObject)
		
		if self.worldPlane == 0:
			#self.gObject.worldPosition = self.parentObject.getPosition() + (-2 * Vector([0, 0, sign * 1]))
			if side == "bottom":
				self.positionRotationX = 180
		elif self.worldPlane == 1:
			#self.gObject.worldPosition = self.parentObject.getPosition() + (-2 * Vector([0, sign * 1, 0]))
			if side == "back":
				self.positionRotationX = 90
			elif side == "front":
				self.positionRotationX = -90
		elif self.worldPlane == 2:
			#self.gObject.worldPosition = self.parentObject.getPosition() + (-2 * Vector([sign * 1, 0, 0]))
			if side == "left":
				self.positionRotationY = -90
			elif side == "right":
				self.positionRotationY = 90
		
		self.gObject.localOrientation = RotationMatrix(self.positionRotationX, 3, "x") * RotationMatrix(self.positionRotationY, 3, "y")

	def setRotation(self):
		"""
		sets the smooth infinite rotation in the right plane of the item object
		"""
		
		perc = (MY_SingleTimer().NOW - self.rotationStart) / self.rotationDuration
		axis = "z"
		
		self.gObject.localOrientation = RotationMatrix(self.positionRotationX, 3, "x") * RotationMatrix(self.positionRotationY, 3, "y") * RotationMatrix(perc * 360, 3, "z")
		
		if perc >= 1:
			self.rotationStart = MY_SingleTimer().NOW

	def setParentStatus(self, has_item=False, item_object=None):
		"""
		sets the staus of the parent object, if it has an item or not and when 'yes', assign the item to it
		
		@type has_item: Boolean
		@param has_item: True if the parent object will have an assigned item
		@type item_object: MY_GameObject
		@param item_object: normally this item (self)
		"""

		self.parentObject.hasItem = has_item
		self.parentObject.item = item_object
		
	def deleteObject(self):
		"""
		deletes object, removes it from the scene and frees its id
		"""

		self.setParentStatus()
		self.gObject.endObject()
		freeId = MY_SingleIdList()
		freeId.setElementId(self.elementId)
		myScene = MY_SingleScene()
		myScene.removeObject(self)

# merch item class
class MY_GameMerchItem(MY_GameObjectActiveHidden):
	"""
	class of the game merchandise items that can be collected by the player
	"""

	def __init__(self, scene, spawn_point):
		"""
		@type scene: KX_Scene
		@param scene: the blender scene object which holds all objects
		@type spawn_point: KX_GameObject
		@param spawn_point: the object where the merch item shall be spawned
		"""
		
		myMerchList = MY_SingleElementList().merchList
		merch_choice = randint(0, len(myMerchList)-1)
		element_name = myMerchList[merch_choice].name
		MY_GameObjectActiveHidden.__init__(self, scene, element_name[2:], spawn_point)
		
		myScene = MY_SingleScene()
		myScene.objectsMerch.append(self)
		
	def setPosition(self, position):
		"""
		set an item at top or bottom of a parented platform
		
		@type position: Vector
		@param position: the vector of the new world position
		"""
		self.gObject.worldPosition = position
	
	def deleteObject(self):
		"""
		deletes object, removes it from the scene and frees its id
		"""
		
		self.gObject.endObject()
		freeId = MY_SingleIdList()
		freeId.setElementId(self.elementId)
		myScene = MY_SingleScene()
		myScene.objectsMerch.remove(self)


# hurdle classhurdleGroundDetector.localOrientation = orientation
class MY_GameHurdle(MY_GameObjectActiveHidden):
	"""
	class of the enemy (hurdle) with its artificial intelligence or something like that ;)

	@type targetPoint: Vector
	@ivar targetPoint: the point the hurdle has to go to
	@type speed: Float
	@ivar speed: the speed of the hurdle
	@type groundObject: MY_GameObject
	@ivar groundObject: the current ground object of the hurdle
	"""
	
	def __init__(self, scene, element_name, spawn_point):
		"""
		@type scene: KX_Scene
		@param scene: the blender scene object which holds all objects
		@type element_name: String
		@param element_name: the name of the hurdle object which shall included into the MY_SingleScene class
		@type spawn_point: KX_GameObject
		@param spawn_point: the object where the merch item shall be spawned
		"""
		
		MY_GameObjectActiveHidden.__init__(self, scene, element_name, spawn_point)
		self.targetPoint = Vector([0, 0, 0])
		self.groundObject = None
		self.speed = 5

		myScene = MY_SingleScene()
		myScene.addObject(self)

	def setPosition(self, position):
		"""
		sets the world position of the hurdle
		
		@type position: Vector
		@param position: the new position of the hurdle
		"""
	
		self.gObject.worldPosition = position

	def setOrientation(self, prota_pos):
		"""
		sets the hurdle z-axis to the world gravity orientation, so the hurdle looks right
		
		@type prota_pos: Vector
		@param prota_pos: the position of the player, where the hurdle looks at
		"""
		
		distance_vector = prota_pos - self.getPosition()
		counter = 0
		look_vector = Vector([0,0])
		for idx, gravity in enumerate(MY_SingleGameWorld().gravityVector):
			if gravity == 0:
				look_vector[counter] = distance_vector[idx]
				counter += 1
		norm_vector_plane = look_vector / math.sqrt(math.pow(look_vector[0],2) + math.pow(look_vector[1],2))
		
		plane_rotation_matrix = RotationMatrix(0,3,"x")
		
		if MY_SingleGameWorld().gravityVector == Vector([0,0,-1]):
			if norm_vector_plane[1] >= 0:
				angle = -(180 / math.pi) * math.acos(norm_vector_plane[0]) - 90
			else:
				angle = (180 / math.pi) * math.acos(norm_vector_plane[0]) - 90
		elif MY_SingleGameWorld().gravityVector == Vector([0,0,1]):
			plane_rotation_matrix = RotationMatrix(180,3,"x")
			if norm_vector_plane[1] >= 0:
				angle = (180 / math.pi) * math.acos(norm_vector_plane[0]) - 90
			else:
				angle = -(180 / math.pi) * math.acos(norm_vector_plane[0]) - 90
		elif MY_SingleGameWorld().gravityVector == Vector([0,-1,0]):
			plane_rotation_matrix = RotationMatrix(90,3,"x")
			if norm_vector_plane[1] >= 0:
				angle = (180 / math.pi) * math.acos(norm_vector_plane[0]) - 90
			else:
				angle = -(180 / math.pi) * math.acos(norm_vector_plane[0]) - 90
		elif MY_SingleGameWorld().gravityVector == Vector([0,1,0]):
			plane_rotation_matrix = RotationMatrix(-90,3,"x")
			if norm_vector_plane[1] >= 0:
				angle = -(180 / math.pi) * math.acos(norm_vector_plane[0]) - 90
			else:
				angle = (180 / math.pi) * math.acos(norm_vector_plane[0]) - 90
		elif MY_SingleGameWorld().gravityVector == Vector([-1,0,0]):
			plane_rotation_matrix = RotationMatrix(-90,3,"y")
			if norm_vector_plane[1] >= 0:
				angle = -(180 / math.pi) * math.acos(norm_vector_plane[0]) + 180
			else:
				angle = (180 / math.pi) * math.acos(norm_vector_plane[0]) + 180
		elif MY_SingleGameWorld().gravityVector == Vector([1,0,0]):
			plane_rotation_matrix = RotationMatrix(90,3,"y")
			if norm_vector_plane[1] >= 0:
				angle = (180 / math.pi) * math.acos(norm_vector_plane[0]) + 180
			else:
				angle = -(180 / math.pi) * math.acos(norm_vector_plane[0]) + 180

		self.gObject.worldOrientation = plane_rotation_matrix * RotationMatrix(angle, 3, "z")
	
	def setHurdle(self, prota_pos):
		"""
		sets the hurdle onto the right way to the player
		
			1. collision sphere (touch) detects items to eat and bullet hits
		
		@type prota_pos: Vector
		@param prota_pos: the position of the object the hurdle is after
		
		@rtype: String
		@return:
			1. 'nothing', if hurdle is up in the air and the 'bullet ohysic engine' takes over
			2. 'item', if hurdle hits an item
			3. 'ground', if hurdle hits the ground
			4. 'dead', if hurdle was hit by a bullet
		"""
		
		# the hurdle collision sphere (is deprecated, maybe it makes errors in the future)
		# can't get the right controller by name so i hope it will be the same sequence all the time
		touch = self.gObject.controllers[2].sensors["hurdle_touch"]
		result = "nothing"
		
		if not self.gObject["fly"]:
			if touch.positive:
				hit_object_name = touch.hitObject.name[2:]
				hit_object = touch.hitObject
			
				if hit_object_name == "delegate_item":
					item = MY_SingleScene().getMyActiveTempObjectById(hit_object["id"])
					item.deleteObject()
					result = "item"

				elif hit_object_name == "platform" or hit_object_name == "single_step" or touch.hitObject.name[2:7] == "stair":
					hitObject = MY_SingleScene().getMyPassiveObjectById(hit_object["id"])
				
					if hitObject != self.groundObject:
						if self.groundObject != None:
							self.groundObject.hurdleObject = None
						self.groundObject = hitObject
						hitObject.hurdleObject = self
					
				
					if hitObject.rootConnectObject != None:
						self.targetPoint = hitObject.rootConnectObject.midPoint
					else:
						self.targetPoint = prota_pos
				
					self.setSpeedToPoint()
				
					result = "ground"
			
				elif hit_object_name == "bullet":
					if hit_object.getLinearVelocity(0) != [0,0,0]:
						print "bullet hit"
						result = "bullet"
				elif hit_object_name == "projectile":
					if hit_object.getLinearVelocity(0) != [0,0,0]:
						#print "projectile hit"
						result = "projectile"
	
		return result
		
	def setSpeedToPoint(self):
		"""
		sets the vectored speed of the hurdle
		"""
	
		speed_vector = Vector([0, 0, 0])
		
		point = self.targetPoint
		
		max_ground_speed = 5
		jump_speed = 7
		
		ground_speed = []
		
		for i,k in enumerate(MY_SingleGameWorld().gravityVector):
			if fabs(k) < 1:
				speed_vector[i] = (point[i] - self.getPosition()[i]) #plane movement
				ground_speed.append([i, speed_vector[i]])
			else:
				speed_vector[i] = k * -jump_speed #jump

		full_ground_speed = fabs(ground_speed[0][1]) + fabs(ground_speed[1][1])
		sign = 1
		if ground_speed[0][1] < 0:
			sign = -1
		speed_vector[ground_speed[0][0]] = fabs(ground_speed[0][1]) / full_ground_speed * sign * max_ground_speed
		sign = 1
		if ground_speed[1][1] < 0:
			sign = -1
		speed_vector[ground_speed[1][0]] = fabs(ground_speed[1][1]) / full_ground_speed * sign * max_ground_speed

		self.gObject.setLinearVelocity(speed_vector, 0)
		
	def deleteObject(self):
		"""
		deletes object, removes it from the scene and frees its id
		"""
		
		if self.groundObject != None:
			self.groundObject.hurdleObject = None
		self.gObject.endObject()
		freeId = MY_SingleIdList()
		freeId.setElementId(self.elementId)
		myScene = MY_SingleScene()
		myScene.removeObject(self)	
		
		
